/*
 X11 routines and image processing code

 The X11 code is from xoids.c, which appparently got it from the X11
 driver in the vogle graphics library

 All other code is copyright (c) N Newell 2002-2004

 Initial version: August 2002

 Updated November 2004 to support 24-bit visuals

*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/times.h>	/* For ticks() */
#include <math.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/keysymdef.h>
#include <X11/xpm.h>

#include "gfxdefs.h"

extern int fps;

#define XP printf

#define SBHEIGHT 17

#define LARGEX11R2	"courier12f.snf"
#define SMALLX11R2	"courier10f.snf"

#define LARGEX11R3	"-adobe-courier-medium-r-normal--24-240-75-75-m-150-iso8859-1"
#define SMALLX11R3	"-adobe-courier-medium-r-normal--10-100-75-75-m-60-iso8859-1"

#define MIN(x,y)	((x) < (y) ? (x) : (y))
#define	CMAPSIZE	256
#define	EV_MASK		KeyReleaseMask|KeyPressMask|ExposureMask|VisibilityChangeMask|StructureNotifyMask

static	Window		winder;
static	Display		*display;
static	int		screen;

static	unsigned long carray[CMAPSIZE];
static	Colormap colormap;

static	Drawable	theDrawable;
static	Pixmap		bbuf;		/* Back buffer pixmap */

static	GC		theGC;
static  GC		unclippedGC;
static  XGCValues	theGCvalues;

static	int		back_used;	/* Have we backbuffered ? */

XRectangle clip_rect;	/* Clipping rectangle for scoreboard */

static	Window oldwin;	/* Window with focus before us */
static int oldrevert;	/* revert_to for old window */

static	XFontStruct	*font_id;
static	XFontStruct	*font_info;

static	unsigned long colour;

static	unsigned int h, w;

static void ndx2rgb(unsigned ndx,int *rp,int *gp,int *bp);

static int lpfilter(int xsize,int ysize,unsigned char *pdata,unsigned xlimit);
static int cliner(int xsize,int ysize,unsigned char *in,unsigned char *out,unsigned xlimit);

/************************* X_init *************************/

/* X_init initialises X11 display.  */

void X_init()
{
int i,r,g,b;
int	x,y,prefx,prefy,prefxs,prefys;
unsigned int bw, depth,mask;
char *av[2],name[50];
Window rootw,childw;
XEvent event;
XSetWindowAttributes theWindowAttributes;
XSizeHints theSizeHints;
unsigned long theWindowMask;
XWMHints theWMHints;
 
/* Get the required window size */

	getxysize(&vdevice.sizeSx,&vdevice.sizeSy);

/* Open the display */

	av[0] = "X11";
	av[1] = (char *)NULL;

	if ((display = XOpenDisplay((char *)NULL)) == (Display *)NULL) {
		fprintf(stderr,"X_init: can't connect to X server\n");
		exit(1);
		}

	winder = DefaultRootWindow(display);
	screen = DefaultScreen(display);
	vdevice.depth = DefaultDepth(display, screen);
	colormap = DefaultColormap(display, screen);

	if (vdevice.depth != 8 && vdevice.depth != 24)
		fprintf(stderr,"WARNING: Unsupported device depth %u (use 8 or 24)\n",vdevice.depth);

/* Set our standard colors... */

	if (vdevice.depth == 1) {		/* Black and white - anything that's not black is white. */
		carray[0] = BlackPixel(display, screen);
		for (i = 1; i < CMAPSIZE; i++)
			carray[i] = WhitePixel(display, screen);
		}
	else {
		X_mapcolor(0, 0, 0, 0);
		X_mapcolor(1, 255, 0, 0);
		X_mapcolor(2, 0, 255, 0);
		X_mapcolor(3, 255, 255, 0);
		X_mapcolor(4, 0, 0, 255);
		X_mapcolor(5, 255, 0, 255);
		X_mapcolor(6, 0, 255, 255);
		X_mapcolor(7, 255, 255, 255);

		for (i=0;i<128;i++) {
/*			ndx2rgb(i*2,&r,&g,&b); /**/
			X_mapcolor(i+8,r,g,b);
			}
		}

//for (i=0;i<7;i++)
// XP("Colour %u: Index %u\n",i,carray[i]);

	XQueryPointer(display, winder, &rootw, &childw, &x, &y, &x, &y, &mask);

	if (childw == None)
		childw = rootw;

	XGetGeometry(display, childw, &rootw, &x, &y, &w, &h, &bw, &depth);

/*
* Tell the window manager not to mess with this window...
* (and hope the bastard of a thing listens)
* Unless it's using prefsize of prefposition of course.
*/
	theWindowAttributes.override_redirect = True;

	prefx = 0;
	prefy = 30; //0;
	prefxs = vdevice.sizeSx;
	prefys = vdevice.sizeSy;;

	if (prefx > -1) {
		theWindowAttributes.override_redirect = False;
	        x = prefx;
	        y = prefy;
		}

	if (prefxs > -1) {
		theWindowAttributes.override_redirect = False;
	        w = prefxs;
	        h = prefys;
		}

	x += bw;
	y += bw;

	w -= 2 * bw;
	h -= 2 * bw;

	/*theWindowMask = CWBackPixel|CWBorderPixel|CWOverrideRedirect;*/
	theWindowMask = CWOverrideRedirect;

//XP("x,y %u %u h,w %u %u\n",x,y,h,w);
	winder = XCreateWindow(display,
			winder,
			x, y,
			w, h,
			bw,
			(int)vdevice.depth,
			InputOutput,
			CopyFromParent,	/* Copy visual from parent window */
			theWindowMask,
			&theWindowAttributes );

	theWMHints.initial_state = NormalState;
	theWMHints.input = True;
	theWMHints.flags = StateHint | InputHint;
	XSetWMHints(display, winder, &theWMHints);
 
	theSizeHints.flags = PPosition|PSize;
	theSizeHints.x = x;
	theSizeHints.y = y;
	theSizeHints.width = w;
	theSizeHints.height = h;

	XSetNormalHints(display, winder, &theSizeHints);

	sprintf(name,"X11 Test");

	XSetStandardProperties(display,
		winder,
		name,
		name,
		None,
		av,
		1,
		&theSizeHints);

	XSelectInput(display, winder, EV_MASK);

	theDrawable = (Drawable)winder;

/* Colors */

	theGCvalues.foreground = WhitePixel(display, screen);
	theGCvalues.background = BlackPixel(display, screen);
	theGCvalues.function = GXcopy;

/* Let us know about the window size. */

	vdevice.sizeX = vdevice.sizeY = MIN(h, w) - 1;
	vdevice.sizeSx = w;
	vdevice.sizeSy = h;

/* Create Graphics Context and Drawable  */

	theGC = XCreateGC(display, RootWindow(display, screen), (GCForeground | GCBackground | GCFunction), &theGCvalues);

	unclippedGC = XCreateGC(display, RootWindow(display, screen), (GCForeground | GCBackground | GCFunction), &theGCvalues);

	X_color(0);
	colour = BLACK;

	XMapRaised(display, winder);
	XFlush(display);

/* Wait for Exposure event.	 */

	do {
		XNextEvent(display, &event);
		} while (event.type != Expose); 

/* Record current input focus, then change it to us */

	XGetInputFocus(display,&oldwin,&oldrevert);
	XSetInputFocus(display,winder,RevertToParent,CurrentTime);

/* Turn off graphics exposures - if left on, a HUGE number of */
/* NoExpose events are generated, slowing things considerably */

	XSetGraphicsExposures(display, theGC, False);
	XSetGraphicsExposures(display, unclippedGC, False);

/* Set keyboard auto-repeat to off */

//	XAutoRepeatOff(display);

/* Set the clipping region = the score box */

	clip_rect.x = 0;
	clip_rect.y = 0; // SBHEIGHT ;
	clip_rect.width = vdevice.sizeSx;
	clip_rect.height = vdevice.sizeSy - SBHEIGHT;

	XSetClipRectangles(display, theGC, 0, 0, &clip_rect, 1, Unsorted);

	font_info = XLoadQueryFont(display, "9x15");

	back_used = 0;

}
/************************* X_exit *************************/

/* cleans up before returning the window to normal. */

void X_exit()
{

	XAutoRepeatOn(display);

	XFlush(display);

	XFreeGC(display, theGC);

	if (back_used) 
		XFreePixmap(display, bbuf);

	XUnmapWindow(display, winder);

	XDestroyWindow(display, winder);

	XSetInputFocus(display,oldwin,RevertToParent,CurrentTime);	/* Restore focus to original window */

}

/* X_clear_key_buffer */

void X_clear_key_buffer()
{
XEvent *event;

	event = (XEvent *) calloc(1, sizeof(XEvent));

	while (XCheckWindowEvent(display, winder, (KeyPressMask | KeyReleaseMask), event));

}

/* X_get_menu_keys */
int X_get_menu_keys()
{
static int r,dl;
XEvent	*event;
char cc[1];
KeySym key;

	event = (XEvent *) calloc(1, sizeof(XEvent));

	if (dl)
		dl--;

	if (r == ENTER)
		r = 0;

	XCheckWindowEvent(display, winder, (KeyPressMask | KeyReleaseMask), event);

	if (event->type == KeyPress) {

		XLookupString((XKeyEvent *) event, &cc[0], 1, &key, NULL);

		dl = 0;

		switch (key) {

		case XK_Up:
		case XK_KP_8:
			r = UPARROW;
			break;

		case XK_Down:
		case XK_KP_2:
			r = DOWNARROW;
			break;

		case XK_Left:
		case XK_KP_4:
			r = LEFTARROW;
			break;

		case XK_Right:
		case XK_KP_6:
			r = RIGHTARROW;
			break;

		case XK_Return:
		case XK_KP_Enter:
			r = ENTER;
			break;

		default:
			break;
		}

	} else if (event->type == KeyRelease) {
		r = 0;
		dl = 0;
	}

	if (dl) {
		return(0);
		}
	else {
		dl = 20;
		return(r);
		}

} /* end X_GET_MENU_KEYS */



/************************* X_check_keypress *************************/

void X_check_keypress()
{
int i,n;
char c[1];
XEvent event;
KeySym 	key;
XComposeStatus compose;

	c[0] = '\0';

	while (XCheckWindowEvent(display, winder, (KeyReleaseMask | KeyPressMask), &event)) {

		n = XLookupString( (XKeyEvent *) &event, &c[0], 1, &key, &compose);

//XP("N= %u Key X = 0x%x\n",n,c[0]);

		switch (event.type) {

		case KeyPress:

			switch (key) {

			case XK_Up:
			case XK_KP_8:
			case XK_R8:
				keyboard_state = keyboard_state | KEY_THRUST1;
				putkey('P'&31);
				break;

			case XK_Down:
			case XK_KP_2:
				keyboard_state = keyboard_state | KEY_THRUST1;
				putkey('N'&31);
				break;

			case XK_Left:
			case XK_KP_4:
			case XK_R10:
				keyboard_state = keyboard_state | KEY_LEFT1;
				putkey('B'&31);
				break;

			case XK_Right:
			case XK_KP_6:
			case XK_R12:
				keyboard_state = keyboard_state | KEY_RIGHT1;	
				putkey('F'&31);
				break;

			case XK_Shift_R:
				keyboard_state = keyboard_state | KEY_FIRE1;
				break;

			case XK_Return:
				putkey('\n');
				break;


			case XK_Escape:
				putkey(0x1b);
				break;

			default:
				if (n>0 && c[0]>0 && c[0]<127)
					putkey(c[0]);
				break;
			}

			break;

		case KeyRelease:

			switch (key) {

			case XK_Up:
			case XK_KP_8:
			case XK_R8:
				keyboard_state = keyboard_state & ~KEY_THRUST1;	
				break;

			case XK_Left:
			case XK_KP_4:
			case XK_R10:
				keyboard_state = keyboard_state & ~KEY_LEFT1;	
				break;

			case XK_Right:
			case XK_KP_6:
			case XK_R12:
				keyboard_state = keyboard_state & ~KEY_RIGHT1;	
				break;

			case XK_Shift_R:
				keyboard_state = keyboard_state & ~KEY_FIRE1;
				break;

			default:
				break;
			}

			break;

		default:

			break;

		}
	}

} /* end X_CHECK_KEYPRESS */



/*
 X_draw

  draws a line from the current graphics position to (x, y).

  Note: (0, 0) is defined as the top left of the window in X (easy to forget).
*/

X_draw(x, y)
int	x, y;
{
	XDrawLine(display,
		theDrawable,
		theGC,
		vdevice.cpVx, vdevice.sizeSy - vdevice.cpVy,
		x, vdevice.sizeSy - y
	);

	XFlush(display);
}

/*
 X_point
	Draw a point!
*/
void X_point(x, y)
int x, y;
{

	XDrawPoint(display, winder, theGC, x, y);

} /* end X_point */



/*
 X_getkey

	grab a character from the keyboard - blocks until one is there.
*/
int X_getkey()
{
	char	c;
	XEvent event;

	do {
		XNextEvent(display, &event);
		if (event.type == KeyPress) {
			if (XLookupString( (XKeyEvent *) &event, &c, 1, NULL, NULL) > 0)
				return((int) c);
			else
				return(0);
		}
	} while (event.type != KeyPress);

	return(0);
}

/************************* X_update *************************/
/*
** X_UPDATE
**
** Flush the X server's queue and handle visilibity and
** window resizing events.
**
*/
void X_update(game_flag)
unsigned int game_flag;
{
XEvent event;

/* Check for expose events */

	if (XCheckWindowEvent(display, winder, VisibilityChangeMask, &event)) {
		X_frontbuf();
		X_clear();
		if (game_flag) {
			X_backbuf();
			}
		}

/* Check for window resize */

	if (XCheckWindowEvent(display, winder, StructureNotifyMask, &event)) {

		if (event.type == ConfigureNotify) {

			vdevice.sizeSx = event.xconfigure.width;
			vdevice.sizeSy = event.xconfigure.height;

			if (vdevice.sizeSx < 320)
				vdevice.sizeSx = 320;

			if (vdevice.sizeSy < 200)
				vdevice.sizeSy = 200;

/* Set the clipping region = the score box */

			clip_rect.x = 0;
			clip_rect.y = 0; // SBHEIGHT;
			clip_rect.width = vdevice.sizeSx;
			clip_rect.height = vdevice.sizeSy - SBHEIGHT;

			XSetClipRectangles(display, theGC, 0, 0, &clip_rect, 1, Unsorted);

/* Change the size of the back buffer */

			XFreePixmap(display, bbuf);

			bbuf = XCreatePixmap(display,
				(Drawable)winder,
				(unsigned int)vdevice.sizeSx,
				(unsigned int)vdevice.sizeSy,
				(unsigned int)vdevice.depth
			);

			X_frontbuf();
			X_clear();
			if (game_flag) {
				X_backbuf();
				X_clear();
			}
		}
	}

/* Use XSYNC so that keyboard is ALWAYS updated! */

	XSync(display, 0);
}

/*
 * X_checkkey
 *
 *	Check if there has been a keyboard key pressed.
 *	and return it if there is.
 */
int X_checkkey()
{
char c;
XEvent event;

	if (!XCheckWindowEvent(display, winder, KeyPressMask, &event))
		return(0);

	if (event.type == KeyPress)
		if (XLookupString( (XKeyEvent *) &event, &c, 1, NULL, NULL) > 0)
			return((int)c);

	return(0);
}

/*
 * X_locator
 *
 *	return the window location of the cursor, plus which mouse button,
 * if any, is been pressed.
 */
int X_locator(wx, wy)
int	*wx, *wy;
{
	Window	rootw, childw;
	int	x, y;
	unsigned int mask;

	XQueryPointer(display, winder, &rootw, &childw, &x, &y, wx, wy, &mask);

	*wy = (int)vdevice.sizeSy - *wy;

	return(mask >> 8);
}

/*
 * X_clear
 *
 * Clear the screen (or current buffer )to current colour
 */
void X_clear()
{
	X_color(BLACK);

	XFillRectangle(display,
		theDrawable,
		theGC,
		0,
		0, 
		vdevice.sizeSx,
		vdevice.sizeSy
		);

}

/*
 * X_color
 *
 *	set the current drawing color index.
 */
void X_color(ind)
int	ind;
{
	colour = carray[ind];
	XSetForeground(display, theGC, colour);
	XSetForeground(display, unclippedGC, colour);
}

/*
 * X_mapcolor
 *
 *	change index i in the color map to the appropriate r, g, b, value.
 */
int X_mapcolor(i, r, g, b)
int	i;
int	r, g, b;
{
int	stat;
XColor tmp;

	if (i >= CMAPSIZE)
		return(-1);

/*
 * For Black and White.
 * If the index is 0 and r,g,b != 0 then we are remapping black.
 * If the index != 0 and r,g,b == 0 then we make it black.
 */

	if (vdevice.depth == 1) {
		if (i == 0 && (r != 0 || g != 0 || b != 0)) 
			carray[i] = WhitePixel(display, screen);
		else if (i != 0 && r == 0 && g == 0 && b == 0)
			carray[i] = BlackPixel(display, screen);
		}
	else {
		tmp.red = (unsigned short)(r / 255.0 * 65535);
		tmp.green = (unsigned short)(g / 255.0 * 65535);
		tmp.blue = (unsigned short)(b / 255.0 * 65535);
		tmp.flags = 0;
		tmp.pixel = (unsigned long)i;

/*printf("Request %5u %5u %5u  ",tmp.red,tmp.green,tmp.blue); /**/

		if ((stat = XAllocColor(display, colormap, &tmp)) == 0) {
			fprintf(stderr, "XAllocColor failed (status = %d)\n", stat);
			exit(1);
			}
/*printf("Got %5u %5u %5u \n",tmp.red,tmp.green,tmp.blue); /**/

		carray[i] = tmp.pixel;
		}

	XFlush(display);
	return(0);
}
	
/*
 * X_font
 *
 *   Set up a hardware font. Return 1 on success 0 otherwise.
 *
 */
X_font(fontfile)
char *fontfile;
{
XGCValues xgcvals;

	if (font_id != (XFontStruct *)NULL)
		XFreeFont(display, font_id);

	if (strcmp(fontfile, "small") == 0) {
		if ((font_id = XLoadQueryFont(display, SMALLX11R2)) == (XFontStruct *)NULL) {		/* X11 R2 */
			if ((font_id = XLoadQueryFont(display, SMALLX11R3)) == (XFontStruct *)NULL)	 	/* X11 R3 */
				return(0);
			else
				fontfile = SMALLX11R3;
		} else
			fontfile = SMALLX11R2;
	} else if (strcmp(fontfile, "large") == 0) {
		if ((font_id = XLoadQueryFont(display, LARGEX11R2)) == (XFontStruct *)NULL) {		/* X11 R2 */
			if ((font_id = XLoadQueryFont(display, LARGEX11R3)) == (XFontStruct *)NULL)	 	/* X11 R3 */
				return(0);
			else
				fontfile = LARGEX11R3;
		} else
			fontfile = LARGEX11R2;
	} else if ((font_id = XLoadQueryFont(display, fontfile)) == (XFontStruct *)NULL)
		return(0);

	vdevice.hheight = font_id->max_bounds.ascent + font_id->max_bounds.descent;
	vdevice.hwidth = font_id->max_bounds.width;

	xgcvals.font = XLoadFont(display, fontfile);
	XChangeGC(display, theGC, GCFont, &xgcvals);

	return(1);
}

/* 
 * X_char
 *
 *	 outputs one char - is more complicated for other devices
 */
X_char(c)
	char	c;
{
	char	*s = " ";

	s[0] = c;
	XDrawString(display, theDrawable, theGC, vdevice.cpVx, (int)(vdevice.sizeSy - vdevice.cpVy), s, 1);
	XFlush(display);
}

/*
 * X_string
 *
 *	Display a string centered at the specified height
 */
void X_string(s, y)
char *s;
int y;
{
int len, wid;

	len = strlen(s);

	wid = XTextWidth(font_info, s, len);
	XDrawString(display, theDrawable, theGC, (int) (vdevice.sizeSx / 2 - wid / 3), y, s, len);

	XSync(display, 0);
}

/*
 * X_fill
 *
 *	fill a polygon
 */
X_fill(n, x, y)
int	n, x[], y[];
{
XPoint	plist[128];
int	i;

	if (n > 128)
		perror("xoids: more than 128 points in a polygon");

	for (i = 0; i < n; i++) {
		plist[i].x = x[i];
		plist[i].y = vdevice.sizeSy - y[i];
		}

	XFillPolygon(display, theDrawable, theGC, plist, n, Nonconvex, CoordModeOrigin);

	vdevice.cpVx = x[n-1];
	vdevice.cpVy = y[n-1];

	XFlush(display);
}

/*
 * X_backbuf
 *
 *	Set up double buffering by allocating the back buffer and
 *	setting drawing into it.
 */
void X_backbuf()
{
	if (!back_used)
		bbuf = XCreatePixmap(display,
			(Drawable)winder,
			(unsigned int)vdevice.sizeSx,
			(unsigned int)vdevice.sizeSy,
			(unsigned int)vdevice.depth
		);

	theDrawable = (Drawable)bbuf;

	back_used = 1;

}


/*
 * X_frontbuf
 *
 *	Make sure we draw to the screen.
 */
void X_frontbuf()
{
	theDrawable = (Drawable)winder;
}

/*
 * X_swapbuf
 *
 *	Swap the back and from buffers. (Really, just copy the
 *	back buffer to the screen).
 */
void X_swapbuf()
{

	XCopyArea(display,
		theDrawable,
		winder,
		theGC,
		0, 17,
		vdevice.sizeSx+1,
		vdevice.sizeSy-17-4-3,
		0, 17
	);

	XSync(display, 0);	/* Not XFlush */
}

/* ================================================== */

/************************* gfxcolumn *************************/

/*
 Draw single column of pixels. Works by creating a one-pixel wide pixmap and
 sending it to the display.
*/

void gfxcolumn(unsigned char *points,int ysize)
{
int i,j,r,g,b,scanbits,bpl;
XImage *xi;
Visual *v;
unsigned char *p,*dp,*s;
unsigned int *ip;

	X_backbuf();

	h = MIN(vdevice.sizeSy,ysize);

	dp = (char *) malloc(h * 4);

	s = points;

/* The colour map array converts the pixel value into the appropriate colour entry */

	if (vdevice.depth<=8) {
		p = dp;
		for (i=0;i<h;i++)
			*p++=carray[*s++];
		}
	else {	/* Assume 24-bit (bad assumption if 16-bit! */
		ip = (unsigned int *) dp;
		for (i=0;i<h;i++) {
			*ip++ = carray[*s++];
			}
		}

/* Get visual and create suitable image */

	v = XDefaultVisual(display,screen);

	scanbits = 8;	/* Assume 8-bits as scanline multiple */
	bpl = 1;		/* Assume one byte per line (single column char pixmap) */

	if (vdevice.depth==24) {
		scanbits = 32;	/* Scanline multiple */
		bpl = 4;	/* Bytes per line */
		}

	xi = XCreateImage(
		display,
		v,				/* Visual */
		vdevice.depth,	/* Depth */
		ZPixmap,		/* Format */
		0,				/* Offset */
		dp,				/* data */
		1,h,			/* Width, Height */
		scanbits,		/* Scanline multiple (bits) */
		bpl);			/* Bytes per line */

/*XP("xoffset %u  format %u  bmunit %u   depth %u  bpl %u  bpp %u  rgb masks %ul %ul %ul\n",
	xi->xoffset,xi->format,xi->bitmap_unit,xi->depth,xi->bytes_per_line,xi->bits_per_pixel,xi->red_mask,xi->green_mask,xi->blue_mask); /**/

	XPutImage(display,theDrawable,theGC,xi,
		0,0,	/* Source offsets */
		0,0,	/* Dest X,Y */
		1,h);	/* X,Y size */

	XDestroyImage(xi);	/* Also frees the image data */
}
/************************* test *************************/
int test(int xsize,int ysize,unsigned char *pdata,int mangle)
{
int i,j,r,g,b,scanbits,bpl;
XImage *xi;
Visual *v;
unsigned char *p,*dp,*op,*s;
unsigned int *ip;

	if (vdevice.depth!=24) {
		printf("Only 24bpp supported!\n");
		return 0;
		}

	X_backbuf();

	h = MIN(vdevice.sizeSy,ysize);
	w = MIN(vdevice.sizeSx,xsize);

	dp = (char *) malloc(h*w*4);	/* Image buffer */
	op = (char *) malloc(h*w*4);	/* Work buffer for image transformations */

	if (dp==NULL || op==NULL) {
		XP("Out of memory\n");
		exit(1);
		}

/* Get data from input image and put it in the image buffer */

	ip = (unsigned int *) dp;
	for (i=0;i<h*w;i++) {
		r=*pdata++;
		g=*pdata++;
		b=*pdata++;
		*ip++ = (r<<16) | (g<<8) | b;
		}

	if (mangle&1 || mangle&16) {
		j = mangle&16;	/* Set for false colour, false for greyscale */
		ip = (unsigned int *) dp;
		for (i=0;i<h*w;i++) {
			r=(*ip>>16)&0xff;
			g=(*ip>>8)&0xff;
			b=(*ip)&0xff;
			recolour(&r,&g,&b,j);
			*ip++ = (r<<16) | (g<<8) | b;
			}
		}

	if (mangle&2) {	/* Horizontal edge detector? */
		if (op != NULL) {
			hedger(w,h,dp,op);
			memcpy(dp,op,h*w*4);
			}
		}

	if (mangle&4) {	/* Vertical edge detector? */
		op = (char *) malloc(h*w*4);
		if (op != NULL) {
			vedger(w,h,dp,op);
			memcpy(dp,op,h*w*4);
			}
		}

	if (mangle&8) {	/* Convolution filter? */
		op = (char *) malloc(h*w*4);
		if (op != NULL) {
			cfilter(w,h,dp,op);
			memcpy(dp,op,h*w*4);
			}
		}

/* Get visual and create suitable image */

	v = XDefaultVisual(display,screen);

	scanbits = 8;	/* Assume 8-bits as scanline multiple */
	bpl = w;		/* Bytes per line = width */

	if (vdevice.depth==24) {
		scanbits = 32;	/* Scanline multiple */
		bpl *= 4;	/* Bytes per line */
		}

	xi = XCreateImage(
		display,
		v,				/* Visual */
		vdevice.depth,	/* Depth */
		ZPixmap,		/* Format */
		0,				/* Offset */
		dp,				/* data */
		w,h,			/* Width, Height */
		scanbits,		/* Scanline multiple (bits) */
		bpl);			/* Bytes per line */

/*XP("xoffset %u  format %u  bmunit %u   depth %u  bpl %u  bpp %u  rgb masks 0x%06x 0x%06x 0x%06x\n",
	xi->xoffset,xi->format,xi->bitmap_unit,xi->depth,xi->bytes_per_line,xi->bits_per_pixel,xi->red_mask,xi->green_mask,xi->blue_mask); /**/

	XPutImage(display,theDrawable,theGC,xi,
		0,0,	/* Source offsets */
		0,0,	/* Dest X,Y */
		w,h);	/* X,Y size */

	XDestroyImage(xi);	/* Also frees the image data */

	free(op);	/* Release our work buffer */

}
/************************* recolour *************************/

static double maxdist=256.0*256.0*256.0;
static int target[3]={0x50,0xb0,0xf0}; 

int set_recolour(unsigned r,unsigned g,unsigned b)
{
int i;

	target[0]=r;
	target[1]=g;
	target[2]=b;

	maxdist = 0.0;

	for (i=0;i<3;i++) {
		if (target[i]>128)
			maxdist += target[i]*target[i];
		else
			maxdist += (255-target[i])*(255-target[i]);
		}

	maxdist=sqrt(maxdist);

/*	XP("r: %u  g:%u   b: %u  Maxdist is %.2lf\n",target[0],target[1],target[2],maxdist);/**/

}
/* Recolour the given pixel using some algorithm.... */

recolour(int *rp,int *gp, int*bp,int mode)
{
int i;
unsigned dist;
double x;

/* Determine pixel RGB colour-space distance from expected colour of grid lines */

	i = *rp-target[0];
	i *= i;

	dist = i;

	i = *gp-target[1];
	i *= i;

	dist += i;

	i = *bp-target[2];
	i *= i;

	dist += i;

	x = dist;

/* Normalise distance (0.0=maximally distant, 1.0=local) */

	x = 1.0-sqrt(x)/maxdist;

//	x=x*x;				/* Square distance to get non-linear mapping */
//	x=x>0.7?1.0:0;		/* Use this to create two-colour image */
//	x=x<0.5?0.0:(x-0.5)*2.0; /* Use this to exagerrate colour distance */

/* Perform the pixel colour mapping */

	if (mode==0) /* Grey-scale based on colour-space distance to selected colour */
		*gp=*rp=*bp = (unsigned) (x*255.0);
	else { /* False-colour */
		i= (unsigned) (x*240.0); /* Maximum scale factor is 255.0 */
		ndx2rgb(i,rp,gp,bp);
		}
}
/************************* hedger *************************/

/* Horizontal edge detector (assumes 4-byte pixels) */

#define HEDGE 100	/* horizontal pixel run filter length */

int hedger(unsigned w,unsigned h,unsigned char *ip,unsigned char *op)
{
register int i,x;
int j,y,k;
double a,b,c,d;
unsigned char *s,*t;

	t = op;		/* Output */
	s = ip;		/* Input */

	for (y=0;y<h;y++) {		/* For each row... */
		for (x=0;x<w;x++) {
			a=b=c=d=0.0;
			for (i=-HEDGE/2;i<HEDGE/2;i++) {	/* Sum HEDGE horizontal pixels */
				if (i+x>=0 && i+x < w) {
					a+=*(s+0+i*4);		/* Treat all fields the same */
					b+=*(s+1+i*4);
					c+=*(s+2+i*4);
					d+=*(s+3+i*4);
					}
				}
			*t++=a/HEDGE;		/* Write output pixel */
			*t++=b/HEDGE;
			*t++=c/HEDGE;
			*t++=d/HEDGE;
			s+=4;				/* Advance input pointer */
			}
		}
}
/************************* vedger *************************/

/* Vertical edge detector (assumes 4-byte pixels) */

#define VEDGE HEDGE	/* vertical pixel run filter length */

int vedger(unsigned w,unsigned h,unsigned char *ip,unsigned char *op)
{
register int i,y;
int j,x,k;
double a,b,c,d;
unsigned char *s,*t,*p;

	for (x=0;x<w;x++) {	/* For each column... */
		t = op+x*4;		/* Output */
		s = ip+x*4;		/* Input */
		for (y=0;y<h;y++) {		/* */
			a=b=c=d=0.0;
			for (i=-VEDGE/2;i<VEDGE/2;i++) {	/* Sum VEDGE vertical pixels */
				if (i+y>=0 && i+y < h) {
					a+=*(s+0+i*4*w);		/* Treat all fields the same */
					b+=*(s+1+i*4*w);
					c+=*(s+2+i*4*w);
					d+=*(s+3+i*4*w);
					}
				}
			*t++=a/VEDGE;		/* Write output pixel */
			*t++=b/VEDGE;
			*t++=c/VEDGE;
			*t++=d/VEDGE;
			t+=(w-1)*4;
			s+=w*4;
			}
		}
}
/************************* cfilter *************************/

/*
 General convolution filter (which assumes 4 byte pixels).

*/

#define XSIZE 21	/* filter length (must be odd if loops are "for (i=-XSIZE/2;i<=XSIZE/2...") */

int cfilter(unsigned w,unsigned h,unsigned char *ip,unsigned char *op)
{
int i,j,k,x,y;
unsigned char *s,*t,*p;
double a,b,c,d;
double *filter,*pf,fsum;

/* Create the filter */

	if ((filter=malloc(XSIZE*XSIZE*sizeof(double))) == NULL) {
		memset(op,0,h*w*4);
		return 0;
		}

	fsum=0.0;
	pf = filter;
	for (y=0;y<XSIZE;y++) {
		for (x=0;x<XSIZE;x++) {
#if 1 /* Generate filter co-efficients */
      /* This code generates a rough detector for vertical/horizontal lines */
			*pf = 0.0; 
			if (x==XSIZE/2)
				*pf=1.0;
			else if (x==XSIZE/2+1 || x==XSIZE/2-1)
				*pf=0.5;
			if (y==XSIZE/2)
				*pf=1.0;
			else if (y==XSIZE/2+1 || y==XSIZE/2-1) {
				if (*pf < 1.0)	/* Don't overwrite X values */
					*pf=0.5;
				}
#else /* Identity filter */
		if (x==XSIZE/2 && y==XSIZE/2)
			*pf=1.0;
		else
			*pf=0.0;
#endif
			fsum+=*pf;
			pf++;
			}
		}

XP("Filter Sum %.2lf  Range %d %d\n",fsum,(int) (-XSIZE/2),(int)(XSIZE/2)); /**/

/* Normalise the filter */

	pf = filter;
	for (y=0;y<XSIZE;y++) {
		for (x=0;x<XSIZE;x++) {
			XP("%3.1lf ",*pf);
			*pf++=*pf/fsum;	/* normalise the filter */
			}
		XP("\n"); /**/
		}

/*
 Filter the input image. Each output pixel is generated from the convolution of the
 input image and the filter kernel created above. Each pixel is treated as being made
 of 4 identical 8-bit fields (usually only 3 of them are actually used, i.e. r,g,b).
*/
	for (y=0;y<h;y++) {		/* For every output row...*/
		for (x=0;x<w;x++) {	/* and  column... */
			a=b=c=d=0.0;	/* Initialise output field values */
			pf = filter;
			for (i=-XSIZE/2;i<=XSIZE/2;i++) {	/* Vertical pixels */
				for (j=-XSIZE/2;j<=XSIZE/2;j++) {	/* Horizontal pixels */
					if (i+y>=0 && i+y < h) {	/* Input row on screen? */
						if (j+x>=0 && j+x < w) {	/* Input column on screen? */
							s = ip+ ((i+y)*w+(j+x))*4;	/* Get pointer to source pixel */
							a+=*pf * *s++;		/* Perform convolution on this subfield */
							b+=*pf * *s++;		/* Repeat for other fields */
							c+=*pf * *s++;
							d+=*pf * *s++;
							}
						}
					pf++;
					}
				}
			t=op+(y*w+x)*4;		/* Compress address of output pixel */

			*t++=a;		/* Write results to output pixel */
			*t++=b;
			*t++=c;
			*t++=d;
			}
		}
}

/************************* screener *************************/
void gfxpaint()
{
int i,dx,dy;

	XFlush(display);

	X_backbuf();	/* Select backbuffer as drawable */

/* Put object on screen */

	XCopyArea(display,theDrawable,winder,theGC,0,0,vdevice.sizeSx,vdevice.sizeSy,0,0);

/* Flush the commands */

	XFlush(display);

	XSetFunction(display,theGC,GXcopy);
}
/*************************  statusbar ************************/


void statusbar(int show)
{
char buff[80];
int i,sby;

	sby = vdevice.sizeSy-SBHEIGHT;

	X_frontbuf();

	X_color(BLACK);

	XFillRectangle(display,theDrawable,unclippedGC,0,sby+1,vdevice.sizeSx,SBHEIGHT);

	if (show) {

		X_color(YELLOW);

#if 0
		sprintf(buff,"FFT's per second: %3u     FFT Size: %u  Slide: %.03lf  Y Scaling: %lf ",
			fps,fftsize(),fftoverlap(),yscaling()/10.0);
#else
		sprintf(buff,"http://hazeii.net/apod.html  mail: hapod@hazeii.net");
#endif
		XDrawString(display, theDrawable, unclippedGC,10,sby+SBHEIGHT-3,buff,strlen(buff));

		XDrawLine(display, theDrawable, unclippedGC, 0,sby, vdevice.sizeSx,sby);

		X_color(BLACK);
		}

	XFlush(display);

	X_backbuf();

}
/************************* ndx2rgb *************************/

/* Convert the given index into false-colour RGB value */

static void ndx2rgb(unsigned ndx,int *rp,int *gp,int *bp)
{
unsigned j;
unsigned char *p;
double x,r,g,b;

	ndx %= 256;	/* Sanity - Force into legal range */

	r = g = b = 0.0;

#if 1 /* Hack...add to offset start position so it doesn't begin with black */
 ndx=(double) ndx/255.0*(255-25)+25;
#endif

	j = ndx/51 * 51;

	x = (double) (ndx-j+1)/51.0;

	switch (j) {

		case 0:	/* Black to blue */
			x = (double) (ndx-j)/50.0;
			r=0.0;
			g=0.0;
			b=x;
			break;

		case 51*1: /* Blue to green */
			b=1.0-x;
			g=x;
			r=0.0;
			break;

		case 51*2:	/* Green to red */
			b=0.0;
			g=1.0-x;
			r=x;
			break;

		case 51*3:	/* Red to Yellow */
			b=0.0;
			g=x*50.0/51.0;
			r=1.0;
			break;

		case 51*4:	/* Yellow to white (note last 10 colours not available) */
			x = (double) (ndx-j)/(51.0-10.0);
			if (x>1.0)
				x=1.0;
			b=x;
			g=1.0;
			r=1.0;
			break;

		case 51*5:
			r = g = b = 1.0;
			break;

		default:
			XP("ERROR: Bad case %u\n",j);
			break;
		}

	*rp = r*255.0;
	*gp = g*255.0;
	*bp = b*255.0;

/* We want to avoid any colours with the same value as system colours, else we don't
   get an identity palette. Maybe there's another way around this, but for now we just
   frig the values to avoid the system colours */

	if (*bp<255)
		 (*bp)++;

	if (*rp<255)
		(*rp)++;

	if (*rp==*gp && *rp==*bp && *rp==255)	/* Avoid 255,255,255 */
		(*bp)--;
}

#if 0
gfxwidth()
{
	return vdevice.sizeSx;
}
gfxheight()
{
	return vdevice.sizeSy;
}
#endif



/************************* bugtest *************************/
int bugtest(int xsize,int ysize,int xlimit,unsigned char *pdata)
{
int i,j,k,imin,imax,scanbits,bpl;
unsigned char *p,*op,*s,*t;
unsigned int delta1,delta2;
double r,g,b,sum,avg;
double min,max,z,scale;
double strip[2000];
FILE *fp;

	h = ysize;
	w = xsize;

XP("Bugtest: Working on image with dimensions %u x %u\n",w,h);
/*for (i=0;i<3*8;i++)
 XP("%02x ",*(pdata+i));
XP("\n");/**/

	op = (char *) malloc(h*w*3);	/* Work buffer for image transformations */

	if (op==NULL) {
		XP("Out of memory\n");
		exit(1);
		}

/* Get data from input image and put it in the image buffer */

	sum = 0.0;
	max = 0.0;
	min = 256.0;

	s=pdata;
	t = op;

/* Compute average value of each horizontal line */

#define XLIMIT w // (xlimit) //1220 /* 'bug' position */

#if 1
	for (i=0;i<h;i++) {

		r = g = b = 0.0;

		for (j=0;j<w;j++) {
			if (j>XLIMIT) {	/* 'bug' is at at right of image */
				s+=3;
				}
			else {
				sum += *s;
#if 1
				k=*s;
				if (k>max)
					max=k;
				if (k<min)
					min=k;
#endif
				r+=*s++;
				g+=*s++;
				b+=*s++;
				}
			}

/*XP("%.8lf\n",r);/**/

		z = r/XLIMIT;

#if 0
		if (z>max)
			max=z;
		if (z<min)
			min=z;
#endif
		strip[i]=z;

		}

	avg = sum/(i*XLIMIT);

	scale = (255-20.0)/(max-min);
	XP("COMPUTED: Avg %.2lf Scale %0.2lf (Min %.2lf Max %.2lf)\n",avg,scale,min,max); /**/

//for (i=0;i<h;i++)
// fprintf(stderr,"%lf\n",strip[i]);
#endif

#if 1
 min=120;
 max=140;
 scale=255/(max-min);
#endif
#if 0
 min=10;max=255;scale=1.0;
#endif
//scale *=4;

	XP("USED: Scale %0.2lf (Min %.2lf Max %.2lf)\n",scale,min,max); /**/

/* Stretch the greyscale */

#if 1
	t = op;
	s = pdata;

	for (i=0;i<h;i++) {
		for (j=0;j<w*3;j++) {
			if (j>XLIMIT*3)
				*t++=*s;
			else {
#if 1
				z = *s;			/* Get pixel from source image */
#else
				z = strip[i];	/* Get average pixel value for this row */
#endif
//				z = ((z-min)*scale)+10.0;
				z = ((z-avg)*scale)+128.0;
				if (z<0)
					z=0;
				if (z>255)
					z=255;
				*t++=z;
				}
			s++;
			}
		}
#else
	memcpy(op,pdata,xsize*ysize*3);
#endif

/* Lowpass filter */

	lpfilter(xsize,ysize,op,xlimit); /**/

/* Centreline */

	cliner(xsize,ysize,op,pdata,xlimit); /**/

/* Squeeze image horizontally */

#if 0
 #define N 10
	for (i=0;i<h;i++) {
		s = t = op+i*w*3;
		for (j=0;j<w/N;j++) {
			z=0;
			for (k=0;k<N*3;k++)
				z+=*s++;
			z/=(N*3);
if (j>w/N-2)
	z=0;
else {
			*t++=z;
			*t++=z;
			*t++=z;
}
			}
		}
#endif

/* Sum vertical pixels across the trail */

#if 0
XP("VPIXEL\n");
	for (j=0;j<w;j++) {
		s = t = op+j*3;
		for (i=0;i<h;i++)
			z+=*(s+i*3);

		z/=h;

		for (i=0;i<h;i++) {
			t = op+j*3+i*w*3;
			*t++=z;
			*t++=z;
			*t++=z;
			}
		}
#endif

/* Sine wave test */

#if 0
	t = op;
	for (i=0;i<h;i++) {
		for (j=0;j<w;j++) {
			z=(double)j/10.0 * 3.14159;
			z=sin(z)*50+128;
			*t++=z;
			*t++=z;
			*t++=z;
			}
		}
#endif


/* Write data to file */

#if 0
	if ((fp = fopen("x.dat","wb")) == NULL)
		XP("File open failed\n");
	else {
		for (j=0;j<h;j++) {	/* j <1 for one pixel strip, or j < h for all */
			for (i=0;i<1024;i++) {	/* i<w or i<1024 for FFT */
				s = op+((j*w+i)*3);
				fwrite(s,1,1,fp);
				}
			}
		fclose(fp);
		XP("File written [%u rows of %u pixels]\n",j,i);
		}
#endif


/* Replace original data with modified info */

	memcpy(pdata,op,h*w*3);

	free(op);	/* Release our work buffer */

}

/* Perform image autocorrelation */

int cortest(double factor,int xsize,int ysize,unsigned char *pdata)
{
unsigned char *op;

	op = (char *) malloc(xsize*ysize*3);	/* Work buffer for image transformations */
	if (op==NULL) {
		XP("Out of memory\n");
		exit(1);
		}

	autocor(factor,xsize,ysize,pdata,op);
	memcpy(pdata,op,h*w*3);
	free(op);	/* Release our work buffer */
}

autocor(double factor,unsigned w,unsigned h,unsigned char *in,unsigned char *op)
{
int i,j,k;
int n,xout;
double xstep,rx,z;
unsigned char *s,*t;

	xout = w/factor;
	xstep = w/factor;

	for (i=0;i<h;i++) {

		s = in+i*w*3;
		t = op+i*w*3;

		for (j=0;j<xout;j++) {

			z=0.0;
			n=0;

			for (rx=j;rx<w;rx+=xstep) {
				k = (int) rx;
				z+=*(s+k*3);
				n++;
				}

			z/=n;

			*t++=z;
			*t++=z;
			*t++=z;
			}
		while (j<w) {
			*t++=255;
			*t++=255;
			*t++=0;
			j++;
			}
		}
}

/************************* lpfilter *************************/

/*
 Crude lowpass filter to remove some of the noise; currently simple sliding-window
 average (worth designing an FIR filter?)
*/

static int lpfilter(int xsize,int ysize,unsigned char *pdata,unsigned xlimit)
{
int i,j,k,m,n,sumcnt;
unsigned char *op,*s,*t;
double x,vsum[2000];

	op = (char *) malloc(xsize*ysize*3);	/* Work buffer for image transformations */
	if (op==NULL) {
		XP("Out of memory\n");
		exit(1);
		}

// 41 used for APOD 'lsfit'
#define LPFLEN 41

	for (i=0;i<ysize;i++) {

		s = pdata+(i*xsize)*3;
		t = op+(i*xsize)*3;

		for (j=0;j<xsize;j++) {

			n=0;
			x=0.0;
			for (k=0;k<LPFLEN;k++) {
				m=j+k-LPFLEN/2;
				if (m>=0 && m<xsize) {
					x+=*(s+m*3);
					n++;
					}
				}
			x /=(double)n;
//fprintf(stderr,"%.8lf\n",x);
			m = x;
//m = *(s+j*3);
			*t++=m;
			*t++=m;
			*t++=m;
			}
//		fprintf(stderr,"\n");
		}

	memcpy(pdata,op,xsize*ysize*3);
	free(op);	/* Release our work buffer */
}
/************************* cliner *************************/

static int cliner(int xsize,int ysize,unsigned char *pdata,unsigned char *op,unsigned xlimit)
{
int i,j,k,m,n;
unsigned char *s,*t;
double x,err,halfw,target[2000],fit[2000];
double z;

/* Target cross-ection - we use a one cycle of a sine wave to simulate the notch */

#define TRKWIDTH 22 /* 22 on APOD 'lsfit' image, TBD(26?) on shear check */

	halfw=ysize/2;
	for (i=0;i<ysize;i++) {
		x=(double) i-ysize/2;	/* +/-ysize/2 */
		if (x<-TRKWIDTH/2 || x >= TRKWIDTH/2)
			x=0.5;
		else {
			x = x/TRKWIDTH*2.0;	/* Range -1 to + 1 */
			x *= 3.14159;
			x =- cos(x)/2+0.5;
			x/=2;
			}
		target[i]=x;
//fprintf(stderr,"%lf\n",x);
		}
//fprintf(stderr,"\n");

#define ALPHA 0.8

	for (j=0;j<xsize;j++) {

		for (i=0;i<ysize/2;i++) {

			s = pdata+(i*xsize+j)*3;	/* Point to start pixel for correlation */

			err=0.0;
			for (k=0;k<ysize/2;k++) {
				x=*(s+k*xsize*3);
				x/=255.0;
				x -= target[k+ysize/4];
				err += x*x;
				}
			fit[i]=err;
/*if (1 || j==300)
 fprintf(stderr,"%lf\n",err); /**/
			}
/*if (1 || j==300)
 fprintf(stderr,"\n"); /**/

		n=0;
		x=fit[0];
		for (i=1;i<ysize/2;i++) {
			if (fit[i]<x) {
				x=fit[i];
				n=i;
				}
			}
		n+=ysize/4;

		t = op+(n*xsize+j)*3;
		*t++=255;
		*t++=255;
		*t++=255;

#if 1
if (j<100) /* Just preset the filter output before the trail start */
 z = 102.5; /* 102.5 on APOD 'lsfit' */
else
 z=z*ALPHA+(double)n*(1.0-ALPHA);
fprintf(stderr,"%u %lf\n",n,z); /**/
#endif

#if 0
		if (n>0) {
			t = pdata+((n-1)*xsize+j)*3;
			*t++=255;
			*t++=0;
			*t++=0;
			}

		if (n<ysize-1) {
			t = pdata+((n+1)*xsize+j)*3;
			*t++=255;
			*t++=0;
			*t++=0;
			}
#endif
		}
memcpy(pdata,op,xsize*ysize*3);
}
